//////////
//
//	File:		ComApplication.c
//
//	Contains:	Application-specific code for basic QuickTime movie display and control.
//				This file is used for BOTH MacOS and Windows.
//
//	Written by:	Tim Monroe
//				Based (heavily!) on the MovieShell code written by Apple DTS.
//
//	Copyright:	 1994-1997 by Apple Computer, Inc., all rights reserved.
//
//	Change History (most recent first):
//
//	   <12>	 	12/16/98	rtm		removed all QTVR API calls, so we can remove QTVR.lib from project
//	   <11>	 	04/02/98	rtm		modified for sprites sample
//	   <10>	 	10/23/97	rtm		moved InitializeQTVR to InitApplication, TerminateQTVR to StopApplication
//	   <9>	 	10/13/97	rtm		reworked HandleApplicationMenu to use menu identifiers
//	   <8>	 	09/11/97	rtm		merged MacApplication.c and WinApplication.c into ComApplication.c
//	   <7>	 	08/21/97	rtm		first file for Windows; based on MacApplication.c for Mac sample code
//	   <6>	 	06/04/97	rtm		removed call to QTVRUtils_IsQTVRMovie in InitApplicationWindowObject
//	   <5>	 	02/06/97	rtm		fixed window resizing code
//	   <4>	 	12/05/96	rtm		added hooks into MacFramework.c: StopApplication, InitApplicationWindowObject
//	   <3>	 	12/02/96	rtm		added cursor updating to DoIdle
//	   <2>	 	11/27/96	rtm		conversion to personal coding style; added preliminary QTVR support
//	   <1>	 	12/21/94	khs		first file
//	   
//////////

// header files

#include "ComApplication.h"
#include "QTSprites.h"

// global variables for Macintosh code
#if TARGET_OS_MAC
Str255						gAppName = "\pQTSprites";				// the name of this application
#endif

long						gMaxMilliSecToUse = 0L;		
	
extern Boolean				gUseBackgroundPicture;


//////////
//
// InitApplication
// Do any application-specific initialization.
//
// The theStartPhase parameter determines which "phase" of application start-up is executed,
// *before* the MDI frame window is created or *after*. This distinction is relevant only on
// Windows, so on MacOS, you should always use kInitAppPhase_BothPhases.
//
//////////

void InitApplication (UInt32 theStartPhase)
{
	// ***do any start-up activities that should occur before the MDI frame window is created
	if (theStartPhase & kInitAppPhase_BeforeCreateFrameWindow) {
	
		// check to make sure that QuickTime wired sprites are available;
		// we depend on these features  
		if (!QTUtils_HasWiredSprites())
			QuitFramework();
		
	}	// end of kInitAppPhase_BeforeCreateFrameWindow
}


//////////
//
// StopApplication
// Do any application-specific shut-down.
//
// The theStopPhase parameter determines which "phase" of application shut-down is executed,
// *before* any open movie windows are destroyed or *after*.
//
//////////

void StopApplication (UInt32 theStopPhase)
{
	// ***do any shut-down activities that should occur after the movie windows are destroyed
	if (theStopPhase & kStopAppPhase_AfterDestroyWindows) {
	
	}	// end of kStopAppPhase_AfterDestroyWindows
}


//////////
//
// DoIdle
// Do any processing that can/should occur at idle time.
//
//////////

void DoIdle (WindowReference theWindow)
{
	WindowObject 		myWindowObject = NULL;
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	myWindowObject = GetWindowObjectFromWindow(theWindow);
	if (myWindowObject != NULL) {
		MovieController		myMC = NULL;
	
		myMC = (**myWindowObject).fController;
		if (myMC != NULL) {

#if TARGET_OS_MAC
			// restore the cursor to the arrow
			// if it's outside the front movie window or outside the window's visible region
			if (theWindow == GetFrontMovieWindow()) {
				Rect	myRect;
				Point	myPoint;
				
				GetMouse(&myPoint);
				MCGetControllerBoundsRect(myMC, &myRect);
				if (!MacPtInRect(myPoint, &myRect) || !PtInRgn(myPoint, GetPortFromWindowReference(theWindow)->visRgn))
					MacSetCursor(&qd.arrow);
			}
#endif	// TARGET_OS_MAC
		}
	}
	
	// @@@INSERT APPLICATION-SPECIFIC IDLE-TIME FUNCTIONALITY HERE
	
	MacSetPort(mySavedPort);
}


//////////
//
// DoUpdateWindow
// Update the specified window.
//
//////////

void DoUpdateWindow (WindowReference theWindow, Rect *theRefreshArea)
{
	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	BeginUpdate(GetPortFromWindowReference(theWindow));
	
	// draw the movie controller and its movie
	MCDoAction(GetMCFromWindow(theWindow), mcActionDraw, theWindow);
	
	EndUpdate(GetPortFromWindowReference(theWindow));
	MacSetPort(mySavedPort);
}


//////////
//
// HandleContentClick
// Handle mouse button clicks in the specified window.
//
//////////

void HandleContentClick (WindowReference theWindow, EventRecord *theEvent)
{
#pragma unused(theEvent)

	GrafPtr 			mySavedPort;
	
	GetPort(&mySavedPort);
	MacSetPort(GetPortFromWindowReference(theWindow));
	
	// @@@INSERT APPLICATION-SPECIFIC CONTENT CLICKING FUNCTIONALITY HERE

	MacSetPort(mySavedPort);
}


//////////
//
// HandleApplicationKeyPress
// Handle application-specific key presses.
// Returns true if the key press was handled, false otherwise.
//
//////////

Boolean HandleApplicationKeyPress (char theCharCode)
{
	Boolean		isHandled = true;
	
	switch (theCharCode) {
	
		// @@@HANDLE APPLICATION-SPECIFIC KEY PRESSES HERE

		default:
			isHandled = false;
			break;
	}

	return(isHandled);
}


#if TARGET_OS_MAC
//////////
//
// CreateMovieWindow
// Create a window to display a movie in.
//
//////////

WindowRef CreateMovieWindow (Rect *theRect, Str255 theTitle)
{
	WindowRef			myWindow;
		
	myWindow = NewCWindow(NULL, theRect, theTitle, false, noGrowDocProc, (WindowPtr)-1L, true, 0);
	return(myWindow);
}
#endif


//////////
//
// HandleApplicationMenu
// Handle selections in the application's menus.
//
// The theMenuItem parameter is a UInt16 version of the Windows "menu item identifier". 
// When called from Windows, theMenuItem is simply the menu item identifier passed to the window proc.
// When called from MacOS, theMenuItem is constructed like this:
// 	*high-order 8 bits == the Macintosh menu ID (1 thru 256)
// 	*low-order 8 bits == the Macintosh menu item (sequential from 1 to ordinal of last menu item in menu)
// In this way, we can simplify the menu-handling code. There are, however, some limitations,
// mainly that the menu item identifiers on Windows must be derived from the Mac values. 
//
//////////

void HandleApplicationMenu (UInt16 theMenuItem)
{
	switch (theMenuItem) {
		case IDM_MAKE_SPRITE_MOVIE:
			QTSprites_CreateSpritesMovie();
			break;
		case IDM_USE_BACKGROUND_IMAGE:
			gUseBackgroundPicture = !gUseBackgroundPicture;
			break;
		default:
			break;
	} // switch (theMenuItem)
}


//////////
//
// AdjustApplicationMenus
// Adjust state of items in the application's menus.
//
// Currently, the Mac application has only one app-specific menu ("Test"); you could change that.
//
//////////

void AdjustApplicationMenus (WindowReference theWindow, MenuReference theMenu)
{
#pragma unused(theWindow)
	MenuReference			myMenu;
	
#if TARGET_OS_WIN32
	myMenu = theMenu;
#elif TARGET_OS_MAC
	myMenu = GetMenuHandle(kTestMenu);
#endif
	
	SetMenuItemCheck(myMenu, IDM_USE_BACKGROUND_IMAGE, gUseBackgroundPicture);
	
	// we don't allow creating new files here...
#if TARGET_OS_MAC
	SetMenuItemState(GetMenuHandle(mFile), iNew, kDisableMenuItem);
#endif
	
}


//////////
//
// DoApplicationEventLoopAction
// Perform any application-specific event loop actions.
//
// Return true to indicate that we've completely handled the event here, false otherwise.
//
//////////

Boolean DoApplicationEventLoopAction (EventRecord *theEvent)
{
	Boolean		isHandled = false;
	
	return(isHandled);
}


//////////
//
// AddControllerFunctionality
// Configure the movie controller.
//
//////////

void AddControllerFunctionality (MovieController theMC)
{
	long			myControllerFlags;
	
	// CLUT table use	
	MCDoAction(theMC, mcActionGetFlags, &myControllerFlags);
	MCDoAction(theMC, mcActionSetFlags, (void *)(myControllerFlags | mcFlagsUseWindowPalette));

	// enable keyboard event handling	
	MCDoAction(theMC, mcActionSetKeysEnabled, (void *)true);
	
	// disable drag support
	MCDoAction(theMC, mcActionSetDragEnabled, (void *)false);
}


//////////
//
// InitApplicationWindowObject
// Do any application-specific initialization of the window object.
//
//////////

void InitApplicationWindowObject (WindowObject theWindowObject)
{
	ApplicationDataHdl		myAppData = NULL;

	if (theWindowObject == NULL)
		return;
		
	myAppData = QTSprites_InitWindowData(theWindowObject);
	(**theWindowObject).fAppData = (Handle)myAppData;
		
}


//////////
//
// RemoveApplicationWindowObject
// Do any application-specific clean-up of the window object.
//
//////////

void RemoveApplicationWindowObject (WindowObject theWindowObject)
{
	if (theWindowObject != NULL)
		QTSprites_DumpWindowData(theWindowObject);

	// DoDestroyMovieWindow in MacFramework.c or MovieWndProc in WinFramework.c
	// releases the window object itself
}


//////////
//
// ApplicationMCActionFilterProc 
// Intercept some mc actions for the movie controller.
//
// NOTE: The theRefCon parameter is a handle to a window object record.
//
//////////

PASCAL_RTN Boolean ApplicationMCActionFilterProc (MovieController theMC, short theAction, void *theParams, long theRefCon)
{
#pragma unused(theMC, theParams)

	Boolean				isHandled = false;
	WindowObject		myWindowObject = NULL;
	
	myWindowObject = (WindowObject)theRefCon;
	if (myWindowObject == NULL)
		return(isHandled);
		
	switch (theAction) {
	
		// handle window resizing
		case mcActionControllerSizeChanged:
			SizeWindowToMovie(myWindowObject);
			break;

		// handle idle events
		case mcActionIdle:
			DoIdle((**myWindowObject).fWindow);
			break;
			
		// handle mouse-down events
		case mcActionMouseDown:
			isHandled = QTSprites_HitTestSprites(myWindowObject, (EventRecord *)theParams);
			break;
			
		default:
			break;
			
	}	// switch (theAction)
	
	return(isHandled);	
}



